import unittest
import six
from .. import server

if six.PY2:
    import mock
else:
    from unittest import mock


class TestServer(unittest.TestCase):

    @mock.patch("perfact.server.safe_syscall")
    def test_getsystemtype(self, mock_syscall):

        # Check devel systems
        mock_syscall.return_value = 0, "mocksystem-9999-devel"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_DEVELOPMENT)

        # Check that we still fetched the pfsystemname
        mock_syscall.assert_called_once_with(['/bin/cat', '/etc/pfsystemname'])

        # Check alternative devel names
        mock_syscall.return_value = 0, "mocksystem-devel-9999"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_DEVELOPMENT)

        # Check validation systems
        mock_syscall.return_value = 0, "mocksystem-9999-validation"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_VALIDATION)

        mock_syscall.return_value = 0, "mocksystem-validation-9999"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_VALIDATION)

        # Check prod systems
        mock_syscall.return_value = 0, "mocksystem-9999-prod"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_PRODUCTION)

        mock_syscall.return_value = 0, "mocksystem-prod-9999"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_PRODUCTION)

        # Check unknown systemtypes
        mock_syscall.return_value = 0, "mocksystem-9999-????"
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, server.PFSYSTEMTYPE_UNKNOWN)

        # Check call error
        mock_syscall.return_value = 1, ""
        ret_val = server.get_pfsysteminfo(kind="type")
        self.assertEqual(ret_val, "unknown")

    @mock.patch("perfact.server.safe_syscall")
    def test_getsystemname(self, mock_syscall):

        testname = "mocksystem-9999-devel"

        # Testcase: Successful return code
        mock_syscall.return_value = 0, testname
        ret_val = server.get_pfsysteminfo(kind="name")

        mock_syscall.assert_called_with(['/bin/cat', '/etc/pfsystemname'])
        self.assertEqual(ret_val, testname)

        # Testcase: Unsuccessful return code
        mock_syscall.return_value = 1, testname
        ret_val = server.get_pfsysteminfo(kind="name")

        self.assertEqual(ret_val, "unknown")

    @mock.patch("perfact.server.safe_syscall")
    def test_getsystemid(self, mock_syscall):

        testid = "1234567890"

        # Testcase: Successful return code
        mock_syscall.return_value = 0, testid
        ret_val = server.get_pfsysteminfo(kind="id")

        mock_syscall.assert_called_with(['/bin/cat', '/etc/pfsystemid'])
        self.assertEqual(ret_val, testid)

        # Testcase: Unsuccessful return code
        mock_syscall.return_value = 1, testid
        ret_val = server.get_pfsysteminfo(kind="id")

        self.assertEqual(ret_val, "unknown")

    def test_getpfsysteminfo_incorrect_kind(self):
        with self.assertRaises(AssertionError):
            server.get_pfsysteminfo(kind="InvalidKind")
